//
//  AnimatedLifeView.m
//  NanoLife
//
//  Created by Scott Stevenson on 2/9/08.
//  Source may be reused with virtually no restriction. See License.txt
//

#import "AnimatedLifeView.h"
#import <QuartzCore/QuartzCore.h>
#import "NSImage-Extras.h"


#pragma mark -
#pragma mark Private Methods and Properties

@interface AnimatedLifeView ()
// holds all glowing sphere layers
@property (retain) CALayer* containerLayerForSpheres;
// the location of the click in a mouse click/drag event
@property CGPoint mouseDownPoint;
@end

@interface AnimatedLifeView (PathAnimations)
// create a new CGPath with a series of random coordinates
- (CGPathRef)newRandomPath;
- (CGPathRef)newRandomPathWithStartingPoint:(CGPoint)firstPoint;
// create a CAAnimation object with result of -newRandomPath as the movement path
- (CAAnimation*)randomPathAnimation;
- (CAAnimation*)randomPathAnimationWithStartingPoint:(CGPoint)firstPoint;
@end

@interface AnimatedLifeView (SphereGeneration)
// create an NSImage to use as the layer contents
- (NSImage*)glowingSphereImageWithScaleFactor:(CGFloat)scale;
// create a new "sphere" layer and add it to the container layer
- (void)generateGlowingSphereLayer;
@end

@interface AnimatedLifeView (Misc)
// create a basic gradient
- (void)setupBackgroundGradient;
// setup colors used to draw glowing sphere images
- (void)setupSphereColors;

- (void)fixCulling;
@end

static double frandom(double start, double end);


#pragma mark -
#pragma mark Main Implementation


@implementation AnimatedLifeView

@synthesize containerLayerForSpheres;
@synthesize countOfSpheresToGenerate;
@synthesize backgroundGradient;
@synthesize mouseDownPoint;
@synthesize sphereCoreColor;
@synthesize sphereGlowColor;
@synthesize fixingCulling;



- (id)initWithFrame:(NSRect)frame
{
    if ( self = [super initWithFrame:frame] )
    {
        // placeholder
    }
    return self;
}

- (void)awakeFromNib
{
    // initialize the random number generator
    srandomdev();
    
    // setup gradient to draw the background of the view
    [self setupBackgroundGradient];
    
    // setup colors used to draw glowing sphere images
    [self setupSphereColors];
        
    // make the view layer-backed and become the delegate for the layer
    self.wantsLayer = YES;
    CALayer* mainLayer = self.layer;
    mainLayer.name = @"mainLayer";
    mainLayer.delegate = self;

    // causes the layer content to be drawn in -drawRect:
    [mainLayer setNeedsDisplay];

    // create container layer for spheres
    CALayer* sphereContainer = [CALayer layer];
    sphereContainer.name = @"sphereContainer";
    [mainLayer addSublayer:sphereContainer];
    self.containerLayerForSpheres = sphereContainer;    

    // generate a set number of spheres
    NSUInteger sphereCount = 80;
    self.countOfSpheresToGenerate = sphereCount;    
    NSUInteger i;
    for ( i = 0; i < sphereCount; i++ )
    {
        [self generateGlowingSphereLayer];        
    }
	
	// Slight pitch
	CATransform3D transform; 
	transform = CATransform3DMakeRotation(-0.2, 1, 0, 0);
	// the value of zDistance affects the sharpness of the transform. 
	float zDistance = 450; 
	transform.m34 = 1.0 / -zDistance;
	mainLayer.sublayerTransform = transform;
	
	self.fixingCulling = false;
}


#pragma mark -
#pragma mark Standard NSView Methods

- (void)drawRect:(NSRect)rect
{
    // draw a basic gradient for the view background
    [self.backgroundGradient drawInRect:self.bounds angle:90.0];
}


- (BOOL)acceptsFirstResponder
{
    // accept keyboard events in the view
    return YES;
}

- (void)keyDown:(NSEvent*)event
{
    // clear all existing layers
    self.containerLayerForSpheres.sublayers = [NSArray array];

    // toggle fullscreen mode
    if ( self.isInFullScreenMode )
        [self exitFullScreenModeWithOptions:nil];
    else
        [self enterFullScreenMode:[NSScreen mainScreen] withOptions:nil];

    // once the screen format changes, reset the spheres
    NSUInteger sphereCount = self.countOfSpheresToGenerate;
    NSUInteger i;
    for ( i = 0; i < sphereCount; i++ )
    {
        [self generateGlowingSphereLayer];        
    }    
}

- (void)mouseDown:(NSEvent*)theEvent
{
    // convert to local coordinate system
    NSPoint mousePointInView = [self convertPoint:theEvent.locationInWindow fromView:nil];

    // convert to CGPoint for convenience
    CGPoint cgMousePointInView = NSPointToCGPoint(mousePointInView);

    // save the original mouse down as a instance variable, so that we
    // can start a new animation from here, if necessary.
    self.mouseDownPoint = cgMousePointInView;
    
    // stop animating everything and move all the sphere layers so that
    // they're directly under the mouse pointer.
    NSArray* sublayers = self.containerLayerForSpheres.sublayers;
    for ( CALayer* layer in sublayers)
    {
        [layer removeAllAnimations];
        layer.position = cgMousePointInView;
    }
	[self fixCulling];
}


- (void)mouseDragged:(NSEvent*)theEvent
{
    // convert to local coordinate system
    NSPoint mousePointInView = [self convertPoint:theEvent.locationInWindow fromView:nil];

    // convert to CGPoint for convenience
    CGPoint cgMousePointInView = NSPointToCGPoint(mousePointInView);
    
    // save the original mouse down as a instance variables, so that we
    // can start a new animation from here, if necessary.
    self.mouseDownPoint = cgMousePointInView;

    [CATransaction begin];

        // make sure the dragging happens immediately. we set a specific
        // value here in case we want to it be nearly instant (0.1) later        
        [CATransaction setValue: [NSNumber numberWithBool:0.0]
                         forKey: kCATransactionAnimationDuration];

        NSArray* sublayers = self.containerLayerForSpheres.sublayers;
        for ( CALayer* layer in sublayers)
        {
            [layer removeAllAnimations];
            layer.position = cgMousePointInView;    
        }

	[self fixCulling];
     [CATransaction commit];
}


- (void)mouseUp:(NSEvent*)anEvent
{
    // start new animation paths for all of the spheres
    NSArray* sublayers = self.containerLayerForSpheres.sublayers;
    for ( CALayer* layer in sublayers )
    {
        // "movementPath" is a custom key for just this app
        CAAnimation* animation = [self randomPathAnimationWithStartingPoint:self.mouseDownPoint];
        [layer addAnimation:animation forKey:@"movementPath"];
    }
	[self fixCulling];
}



#pragma mark -
#pragma mark Path Animations

// ---------------------------------------------------------------------------
// -newRandomPath
// ---------------------------------------------------------------------------
// create a new CGPath with a series of random coordinates

- (CGPathRef)newRandomPath
{
    CGPoint point;
    NSSize size = self.bounds.size;
    NSInteger x = random() % ((NSInteger)(size.width) + 50);
    NSInteger y = random() % ((NSInteger)(size.height) + 50);
    point = CGPointMake( x, y );
    
    return [self newRandomPathWithStartingPoint:point];
}


// ---------------------------------------------------------------------------
// -newRandomPathWithStartingPoint:
// ---------------------------------------------------------------------------
// create a new CGPath with a series of random coordinates

- (CGPathRef)newRandomPathWithStartingPoint:(CGPoint)firstPoint
{
    // create an array of points, with 'firstPoint' at the first index
    NSUInteger count = 10;
    CGPoint onePoint;
    CGPoint allPoints[count];    
    allPoints[0] = firstPoint;

    // create several CGPoints with random x and y coordinates
    CGMutablePathRef thePath = CGPathCreateMutable();
    NSUInteger i;
    for ( i = 1; i < count; i++)
    {
        // allow the coordinates to go slightly out of the bounds of the view (+50)
        NSInteger x = random() % ((NSInteger)(self.bounds.size.width) + 50);
        NSInteger y = random() % ((NSInteger)(self.bounds.size.height) + 50);
        onePoint = CGPointMake(x,y);
        allPoints[i] = onePoint;
    }

    CGPathAddLines ( thePath, NULL, allPoints, count );     
    return thePath;
}


// ---------------------------------------------------------------------------
// -randomPathAnimation
// ---------------------------------------------------------------------------
// create a CAAnimation object with result of -newRandomPath as the movement path

- (CAAnimation*)randomPathAnimation
{
    CGPoint point;
    NSSize size = self.bounds.size;
    NSInteger x = random() % ((NSInteger)(size.width) + 50);
    NSInteger y = random() % ((NSInteger)(size.height) + 50);
    point = CGPointMake( x, y );  
      
    return [self randomPathAnimationWithStartingPoint:point];
}


// ---------------------------------------------------------------------------
// -randomPathAnimationWithStartingPoint:
// ---------------------------------------------------------------------------
// create a CAAnimation object with result of -newRandomPath as the movement path

- (CAAnimation*)randomPathAnimationWithStartingPoint:(CGPoint)firstPoint
{
    CGPathRef path = [self newRandomPathWithStartingPoint:firstPoint]; 
    CAKeyframeAnimation* animation;
    
    animation                   = [CAKeyframeAnimation animationWithKeyPath:@"position"];
    animation.path              = path;
    animation.timingFunction    = [CAMediaTimingFunction functionWithName:kCAMediaTimingFunctionEaseOut];
    animation.duration          = ( random() % 32 + 15 ); // 15-32 seconds
    animation.autoreverses      = YES;
    animation.repeatCount       = 1e100;

    CGPathRelease ( path );        
    return animation;
}



#pragma mark -
#pragma mark Sphere Layer Generation


// ---------------------------------------------------------------------------
// -glowingSphereImageWithScaleFactor:
// ---------------------------------------------------------------------------
// create a new "sphere" layer and add it to the container layer

- (NSImage*)glowingSphereImageWithScaleFactor:(CGFloat)scale
{
    if ( scale > 10.0 || scale < 0.5 ) {
        NSLog(@"%s: larger than 10.0 or less than 0.5 scale. returning nil.", _cmd);
        return nil;        
    }
    
    // the image is two parts: a core sphere and a blur.
    // the blurred image is larger, and the final image
    // must be large enough to contain it.
    NSSize sphereCoreSize = NSMakeSize(5*scale,5*scale);    
    NSSize sphereBlurSize = NSMakeSize(10*scale,10*scale);
    NSSize finalImageSize = NSMakeSize(sphereBlurSize.width*2,sphereBlurSize.width*2);
    NSRect finalImageRect;
    finalImageRect.origin = NSZeroPoint;
    finalImageRect.size   = finalImageSize;

    // define a drawing rect for the core of the sphere
    NSRect sphereCoreRect;
    sphereCoreRect.origin = NSZeroPoint;
    sphereCoreRect.size = sphereCoreSize;
    NSColor* coreColor = self.sphereCoreColor;
    CGFloat sphereCoreOffset = (finalImageSize.width - sphereCoreSize.width) * 0.5;
    
    // create the "core sphere" image
    NSImage* solidCircle = [[NSImage alloc] initWithSize:sphereCoreSize];
    [solidCircle lockFocus];
        [coreColor setFill];
        [[NSBezierPath bezierPathWithOvalInRect:sphereCoreRect] fill];
    [solidCircle unlockFocus];
    
    // define a drawing rect for the sphere blur
    NSRect sphereBlurRect;
    sphereBlurRect.origin.x = (finalImageSize.width - sphereBlurSize.width) * 0.5;
    sphereBlurRect.origin.y = (finalImageSize.width - sphereBlurSize.width) * 0.5;
    sphereBlurRect.size = sphereBlurSize;

    // create the "sphere blur" image (not yet blurred)
    NSImage* blurImage = [[NSImage alloc] initWithSize:finalImageSize];
    NSColor* blurColor = self.sphereGlowColor;
    [blurImage lockFocus];
        [blurColor setFill];
        [[NSBezierPath bezierPathWithOvalInRect:sphereBlurRect] fill];
    [blurImage unlockFocus];
    
    // convert the "sphere blur" image to a CIImage for processing
    NSData* dataForBlurImage = [blurImage TIFFRepresentation];
    CIImage* ciBlurImage = [CIImage imageWithData:dataForBlurImage];

    // apply the blur using CIGaussianBlur
    CIFilter* filter = [CIFilter filterWithName:@"CIGaussianBlur"];
    [filter setDefaults];
    NSNumber* inputRadius = [NSNumber numberWithFloat:3.0];
    [filter setValue:inputRadius forKey:@"inputRadius"];
    [filter setValue:ciBlurImage forKey:@"inputImage"];
    CIImage* ciBlurredImage = [filter valueForKey:@"outputImage"];
    ciBlurredImage = [ciBlurredImage imageByCroppingToRect:NSRectToCGRect(finalImageRect)];

    // draw the final image
    NSImage* compositeImage = [[NSImage alloc] initWithSize:finalImageSize];
    [compositeImage lockFocus];    
    
        // draw glow first
        [ciBlurredImage drawInRect:finalImageRect
                          fromRect:finalImageRect
                         operation:NSCompositeSourceOver
                          fraction:0.7];
                          
        // now draw solid sphere on top
        [solidCircle drawInRect:NSOffsetRect(sphereCoreRect,sphereCoreOffset,sphereCoreOffset)
                       fromRect:sphereCoreRect
                      operation:NSCompositeSourceOver
                       fraction:1.0];
                        
    [compositeImage unlockFocus];
    
    [solidCircle release];
    [blurImage release];
    
    return [compositeImage autorelease];
}


// ---------------------------------------------------------------------------
// -generateGlowingSphereLayer
// ---------------------------------------------------------------------------
// create a new "sphere" layer and add it to the container layer

- (void)generateGlowingSphereLayer
{
    // generate a random size scale for glowing sphere
    NSUInteger randomSizeInt = (random() % 200 + 50 );
    CGFloat sizeScale = (CGFloat)randomSizeInt / 100.0;    
    NSImage* compositeImage = [self glowingSphereImageWithScaleFactor:sizeScale];
    CGImageRef cgCompositeImage = [compositeImage cgImage];
    
    // generate a random opacity value with a minimum of 15%
    NSUInteger randomOpacityInt = (random() % 100 + 15 );
    CGFloat opacityScale = (CGFloat)randomOpacityInt / 100.0;
    
    CALayer* sphereLayer            = [CALayer layer];
    sphereLayer.name                = @"glowingSphere";
    sphereLayer.bounds              = CGRectMake ( 0, 0, 20, 20 );
    sphereLayer.contents            = (id)cgCompositeImage;
    sphereLayer.contentsGravity     = kCAGravityCenter;
    sphereLayer.delegate            = self;    
    sphereLayer.opacity             = opacityScale;  
	sphereLayer.zPosition			= frandom(250, -250);
    
    // "movementPath" is a custom key for just this app
    [self.containerLayerForSpheres addSublayer:sphereLayer];        
    [sphereLayer addAnimation:[self randomPathAnimation] forKey:@"movementPath"];
    
    CGImageRelease ( cgCompositeImage );
}



#pragma mark -
#pragma mark Misc

// ---------------------------------------------------------------------------
// -setupBackgroundGradient
// ---------------------------------------------------------------------------
// create a basic gradient

- (void)setupBackgroundGradient
{
    // create a basic gradient for the background of the view
    
    CGFloat red1   =    0.0 / 255.0;
    CGFloat green1 =   72.0 / 255.0;
    CGFloat blue1  =  127.0 / 255.0;

    CGFloat red2    =   0.0 / 255.0;
    CGFloat green2  =  43.0 / 255.0;
    CGFloat blue2   =  76.0 / 255.0;

    NSColor* gradientTop    = [NSColor colorWithCalibratedRed:red1 green:green1 blue:blue1 alpha:1.0];    
    NSColor* gradientBottom = [NSColor colorWithCalibratedRed:red2 green:green2 blue:blue2 alpha:1.0];

    NSGradient* gradient;
    gradient = [[NSGradient alloc] initWithStartingColor:gradientBottom endingColor:gradientTop];

    self.backgroundGradient = gradient;
    [gradient release];
}


// ---------------------------------------------------------------------------
// -setupSphereColors
// ---------------------------------------------------------------------------
// setup colors used to draw glowing sphere images

- (void)setupSphereColors
{
    self.sphereCoreColor = [NSColor whiteColor];
    
    CGFloat red   =   0.0 / 255.0;
    CGFloat green = 189.0 / 255.0;
    CGFloat blue  = 255.0 / 255.0;
    
    self.sphereGlowColor = [NSColor colorWithCalibratedRed:red green:green blue:blue alpha:1.0];
}


// Make a layer take up the whole screen
- (void)	fixCulling
{
    CALayer* layer = [self.containerLayerForSpheres.sublayers objectAtIndex:0];
	if (self.fixingCulling)
	{
		[layer removeAllAnimations];
		layer.opacity = 0.01;
		layer.transform = CATransform3DMakeScale(400, 400, 1);
	}
	else
	{
		layer.opacity = 1;
		layer.transform = CATransform3DIdentity;
	}
}

// Take value from checkbox
- (IBAction)fixCulling:(id)sender
{
	self.fixingCulling = [sender state];
}


static double frandom(double start, double end)
{
  double r = random();
  r /= RAND_MAX;
  r = start + r*(end-start);
  
  return r;
}

@end
